#####################################
# Cognitive miserliness and argument literacy: 
# Effects of intuitive and analytic thinking on recognizing fallacies
#####################################

rm(list=ls())
graphics.off()

# --------------- Read data, clean it up, and prepare indices ------------------

d0 <- read.csv("rawdata.csv", encoding="UTF-8")
##  [1] "subject"        "trial"          "stimfile"       "condition"      "code"           "resp.1"        
##  [7] "resp.2"         "response"       "distractor"     "error"          "init.time"      "RT"            
## [13] "MD.1"           "MD.2"           "AUC.1"          "AUC.2"          "MD.time"        "x.flip"        
## [19] "y.flip"         "z.MD.separate"  "z.MD.together"  "z.AUC.separate" "z.AUC.together" "comments"      
## 25-125 X values; 126-226 y values
## 58 Ss, 120 trials each
## init.time is time to first movement, RT is the total

# Extract item numbers from item_number_key.csv and add them to d0
data_key <- read.csv("item_number_key.csv", encoding="UTF-8", header=FALSE, sep=";")
names(data_key) <- c("item_nr", "orig_fin", "label_eng")
rownames(data_key) <- trimws(data_key$orig_fin)
d0$item_nr <- data_key[trimws(d0$stimfile),"item_nr"]

pure_fallacies <- paste("fal",c(1,2,3,4,5,6,7,8,9,11,14,15,16,17,18,19,20,21,22,
                                23,24,25,26,28,30,31,32,33,34,35,36,37,39,40,41,
                                42,43,44,45,46,47,48,50,51,52,53,54,55,56,59,61,
                                62,63), sep="")
pure_reasonable <- paste("ok", c(2,4,7,9,10,11,12,13,15,16,19,21,24,28,29,31,33,
                                 34,35,36,39,43,44,45,46,47,49,56,57), sep="")

# Select only unambiguous items (identified in separate pilot) for analysis
d0 <- d0[d0$item_nr %in% c(pure_fallacies, pure_reasonable),] 

# Indices 
S <- d0$subject
T1 <- d0$trial<65 
T2 <- d0$trial>64
fal <- d0$item_nr %in% pure_fallacies
rea <- d0$item_nr %in% pure_reasonable


# Participants were assigned to four different combinations of manipulation and 
# side of response buttons in rotating order as they arrived at the lab.
# Thus the two first were intuitive, the two next analytic, etc.
Analytic <- d0[,1] %in% c(3,4,7,8,11,12,15,16,19,20,23,24,27,28,31,32,35,36,
                          39,40,43,44,47,48,51,52,55,56) 
Analytic.s <- tapply(Analytic,S,mean) # logical: 0= int, 1= analytic

# ---------------------- Prepare dependent variables ---------------------------

# DV 1) SDT measures
#####################

library(psycho)

# Count the numbers of hits, misses, false alarms and correct rejections  
miss_rea_bef <- c()
hit_rea_bef <- c()
fa_fal_bef <- c()
cr_fal_bef <- c()
miss_rea_aft <- c()
hit_rea_aft <- c()
fa_fal_aft <- c()
cr_fal_aft <- c()

for (i in unique(S)) {
  miss_rea_bef[i] <- sum(S==i & d0$error==1 & T1 & rea);
  hit_rea_bef[i] <- sum(S==i & d0$error==0 & T1 & rea); 
  fa_fal_bef[i] <- sum(S==i & d0$error==1 & T1 & fal); 
  cr_fal_bef[i] <- sum(S==i & d0$error==0 & T1 & fal);
  miss_rea_aft[i] <- sum(S==i & d0$error==1 & T2 & rea);
  hit_rea_aft[i] <- sum(S==i & d0$error==0 & T2 & rea); 
  fa_fal_aft[i] <- sum(S==i & d0$error==1 & T2 & fal); 
  cr_fal_aft[i] <- sum(S==i & d0$error==0 & T2 & fal) }

SDT <- data.frame(subject_nr =sort(unique(S)),
                  miss_rea_bef, hit_rea_bef, fa_fal_bef, cr_fal_bef,
                  miss_rea_aft, hit_rea_aft, fa_fal_aft, cr_fal_aft)

# Before manipulation
SDTbefore <- dprime(n_fa = SDT$fa_fal_bef, 
                    n_hit = SDT$hit_rea_bef, 
                    n_miss = SDT$miss_rea_bef, 
                    n_cr = SDT$cr_fal_bef,
                    adjusted=TRUE,
                    n_targets = SDT$hit_rea_bef + SDT$miss_rea_bef,
                    n_distractors = SDT$fa_fal_bef + SDT$cr_fal_bef)

dprime1 <- SDTbefore$dprime
c1 <- SDTbefore$c

# After manipulation
SDTafter <- dprime(n_fa = SDT$fa_fal_aft, 
                   n_hit = SDT$hit_rea_aft, 
                   n_miss = SDT$miss_rea_aft, 
                   n_cr = SDT$cr_fal_aft,
                   adjusted=TRUE,
                   n_targets = SDT$hit_rea_aft + SDT$miss_rea_aft,
                   n_distractors = SDT$fa_fal_aft + SDT$cr_fal_aft)

dprime2 <- SDTafter$dprime
c2 <- SDTafter$c


# DV 2: Accuracy (error rates)
##############################
error.s <- tapply(d0$error,S,mean) # mean of each participant
error1.s <- tapply(d0$error[T1],S[T1],mean) 
error2.s <- tapply(d0$error[T2],S[T2],mean)  
error_fal_1 <- tapply(d0$error[T1 & fal], S[T1 & fal], mean) 
error_fal_2 <- tapply(d0$error[T2 & fal], S[T2 & fal], mean) 
error_rea_1 <- tapply(d0$error[T1 & rea], S[T1 & rea],mean) 
error_rea_2 <- tapply(d0$error[T2 & rea], S[T2 & rea], mean) 


# DV 3: RT 
############
RT <- log(d0$RT) # a vector with the RT of each trial of each participant
RT.s <- tapply(RT,S,mean) 
RT1.s <- tapply(RT[T1],S[T1],mean) 
RT2.s <- tapply(RT[T2],S[T2],mean) 
RT_cor <- tapply(RT[d0$error==0], S[d0$error==0],mean)
RT_err <- tapply(RT[d0$error==1], S[d0$error==1],mean)



# Mouse tracking measures
# --------------------------

# DV 4: Xrat
# ############

# Preprocessing: Turn all trajectories in the same direction. 
# Positive values = side of final response.  
Side <- sign(d0[,125]) # side on which response was made 1=right, -1=left
Xpos <- as.matrix(d0[,25:125]*Side) # -1 to 1; start is 0, 1 now means on side of response

Ypos <- as.matrix(d0[,126:226]) # 0 to 1.5; start is 0

Xsame <- rowSums(pmax(Xpos,0)) # how much mouse stayed on side of response
Xopposite <- rowSums(pmax(-Xpos,0)) # how much the mouse visited the other side
Xrat <- Xopposite/(Xsame+Xopposite) # proportion of time and severity of "other side"      

Xrat.s <- tapply(Xrat,S,mean)     # mean Xrat for each participant
Xrat1.s <- tapply(Xrat[T1],S[T1],mean) # before
Xrat2.s <- tapply(Xrat[T2],S[T2],mean) # after
Xrat_cor <- tapply(Xrat[d0$error==0], S[d0$error==0],mean)
Xrat_err <- tapply(Xrat[d0$error==1], S[d0$error==1],mean)



# DV 5: AUC
# ############

# Rawdata has AUC in separate columns depending on where the response buttons 
# were (left or right). Combine the columns into one variable:
AUC <- ifelse(is.na(d0$AUC_1),d0$AUC_2,d0$AUC_1)  
AUC.s <- tapply(AUC,S,mean)  
AUC1.s <- tapply(AUC[T1],S[T1],mean) 
AUC2.s <- tapply(AUC[T2],S[T2],mean) 
AUC_cor <- tapply(AUC[d0$error==0], S[d0$error==0],mean)
AUC_err <- tapply(AUC[d0$error==1], S[d0$error==1],mean)


# ------------------------------- RESULTS -------------------------------------


# Are the different dependent measures associated with each other across trials?
DVs <- cbind(d0$error, RT, Xrat, AUC)
corr.test(DVs)
#       error RT   Xrat  AUC
# error 1.00  0.20 0.13 0.13
# RT    0.20  1.00 0.23 0.17
# Xrat  0.13  0.23 1.00 0.48
# AUC   0.13  0.17 0.48 1.00
# Sample Size 4756, all p's <.001 (adjusted for multiple tests) 


# ------------------------------- TABLE 1 -------------------------------------

# Change scores in the studied variables by type of manipulation and type of 
# argument

change_dprime <- dprime2 - dprime1
change_crit <- c2 - c1
change_error <- error2.s - error1.s
change_RT <- RT2.s - RT1.s
change_Xrat <- Xrat2.s - Xrat1.s
change_AUC <- AUC2.s - AUC1.s

library(psych)
changes <- cbind(change_dprime, change_crit, change_error, change_RT, 
                 change_Xrat, change_AUC)
describe.by(changes, group= Analytic.s, skew=FALSE) 

" Descriptive statistics by group 
INDICES: 0 = intuitive group
                 vars  n  mean   sd   min   max range   se
change_dprime       1 30 -0.13 0.83 -1.63  1.77  3.40 0.15
change_crit         2 30 -0.23 0.43 -1.11  0.62  1.73 0.08
change_error        3 30  0.03 0.09 -0.17  0.18  0.35 0.02
change_RT           6 30 -0.38 0.18 -0.75 -0.08  0.66 0.03
change_Xrat         7 30  0.01 0.07 -0.11  0.21  0.32 0.01
change_AUC          8 30  0.10 0.21 -0.31  0.45  0.76 0.04
----------------------------------------------------------- 
INDICES: 1 = analytic
                 vars  n  mean   sd   min  max range   se
change_dprime       1 28  0.03 0.86 -1.33 1.54  2.88 0.16
change_crit         2 28 -0.14 0.41 -0.97 0.54  1.51 0.08
change_error        3 28  0.00 0.08 -0.15 0.14  0.28 0.02
change_RT           6 28  0.06 0.19 -0.33 0.54  0.86 0.04
change_Xrat         7 28  0.01 0.06 -0.12 0.12  0.24 0.01
change_AUC          8 28 -0.01 0.22 -0.62 0.57  1.19 0.04 
"

# ------------------- ANALYSES OF GROUP DIFFERENCES ---------------------------

# Investigate whether size of change from baseline to after manipulation 
# differed between those receiving intuitive and analytic manipulations

# DV 1) SDT measures  
-----------------------
t.test((dprime2 - dprime1)~Analytic.s)
# t = -0.75029, df = 55.296, p-value = 0.4563
# mean in group 0 mean in group 1 
# -0.13278857      0.03372207

t.test((c2 - c1)~Analytic.s)
# t = -0.88562, df = 55.916, p-value = 0.3796
# mean in group 0 mean in group 1 
# -0.2340326      -0.1366800 


# DV 2) Accuracy (error rates) 
# -----------------------------
t.test((error2.s - error1.s)~Analytic.s)
# t = 0.95679, df = 55.904, p-value = 0.3428
# mean in group 0 mean in group 1 
# 0.025456109     0.003429467 


# DV 3) RT                                                      
# ----------------------------
t.test((RT2.s - RT1.s)~Analytic.s)
# t = -9.1187, df = 54.674, p-value = 1.442e-12       ***
# mean in group 0 mean in group 1 
# -0.3796321       0.0623938


# DV 4) Xrat and number of changes of mind
# ----------------------------
t.test((Xrat2.s - Xrat1.s)~Analytic.s)         
# t = -0.13786, df = 55.545, p-value = 0.8909
# mean in group 0 mean in group 1 
# 0.007857867     0.010114706 


# DV 5) AUC 
# -------------------------
t.test((AUC2.s - AUC1.s)~Analytic.s)     
# t = 2.0129, df = 55.247, p-value = 0.04901         *
# mean in group 0 mean in group 1 
# 0.09962569     -0.01374567 



# Do measures differ between trials ending in correct responses or errors?
# -------------------------------------------------------------------------
describe(cbind(RT_cor, RT_err, Xrat_cor, Xrat_err, AUC_cor, AUC_err), skew=FALSE)

"
        vars  n mean   sd   min   max range   se
RT_cor      1 58 8.96 0.22  8.47  9.38  0.91 0.03
RT_err      2 58 9.38 0.32  8.65 10.09  1.44 0.04
Xrat_cor    3 58 0.12 0.08  0.01  0.36  0.36 0.01
Xrat_err    4 58 0.22 0.14  0.00  0.71  0.71 0.02
AUC_cor     5 58 0.23 0.19  0.00  0.78  0.78 0.02
AUC_err     6 58 0.54 0.38 -0.04  1.39  1.43 0.05
"

t.test(RT_cor, RT_err, paired=TRUE) # t = -11.234, df = 57, p-value = 4.455e-16
t.test(Xrat_cor, Xrat_err, paired=TRUE) # t = -5.7054, df = 57, p-value = 4.347e-07
t.test(AUC_cor, AUC_err, paired=TRUE) # t = -6.5733, df = 57, p-value = 1.63e-08

# All mouse tracking measures are larger for error trials. Indicates more 
# attraction towards the other response when making errors than when responding
# correctly.

corr.test(Xrat_cor, Xrat_err) # r = .42
# ------------------------------ FIGURE 1 -------------------------------------- 

# X axis shows the two times: before and after manipulations
x = c(1,2)

# Stadard error of mean, needed for error bars in Figure 1
se <- function(x) sd(x)/sqrt(length(x))

# Figure1a() draws a line graph of the means of the studied measures across time 
# (before and after manipulation), type of manipulation used, and argument type.

Figure1a <- function(fal1, fal2, rea1, rea2, paneltitle, ylabel, ylimits) {
  # Lines showing means
  plot(x, c(mean(fal1[Analytic.s==0]),  mean(fal2[Analytic.s==0])), 
       type = "o", col="red", lty=1,
       lwd=3, pch=0, cex=1.5, xlab="Time", 
       ylab=ylabel, main=paneltitle, xaxt="none", xlim=c(0.9, 2.1), 
       ylim=ylimits); 
  axis(1, at=c(1,2), labels=c("Before", "After"));
  lines(x, c(mean(rea1[Analytic.s==0]), 
             mean(rea2[Analytic.s==0])), 
      type = "o", col="red", lty=2,
      lwd=3, pch=1, cex=1.5);
  lines(x, c(mean(fal1[Analytic.s==1]), 
             mean(fal2[Analytic.s==1])), 
      type = "o", col="blue",lty=1,
      lwd=3, pch=2, cex=1.5);
  lines(x, c(mean(rea1[Analytic.s==1]), 
             mean(rea2[Analytic.s==1])), 
      type = "o", col="blue", lty=2,
      lwd=3, pch=6, cex=1.5);
  # Error bars in the before condition
  arrows(x0=1, 
         y0=mean(fal1[Analytic.s==0])-se(fal1[Analytic.s==0]), 
         y1=mean(fal1[Analytic.s==0])+se(fal1[Analytic.s==0]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="red");
  arrows(x0=1, 
         y0=mean(rea1[Analytic.s==0])-se(rea1[Analytic.s==0]), 
         y1=mean(rea1[Analytic.s==0])+se(rea1[Analytic.s==0]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="red");
  arrows(x0=1, 
         y0=mean(fal1[Analytic.s==1])-se(fal1[Analytic.s==1]), 
         y1=mean(fal1[Analytic.s==1])+se(fal1[Analytic.s==1]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="blue");
  arrows(x0=1, 
         y0=mean(rea1[Analytic.s==1])-se(rea1[Analytic.s==1]),
         y1=mean(rea1[Analytic.s==1])+se(rea1[Analytic.s==1]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="blue");
  # Error bars in the after condition
  arrows(x0=2, 
         y0=mean(fal2[Analytic.s==0])-se(fal2[Analytic.s==0]), 
         y1=mean(fal2[Analytic.s==0])+se(fal2[Analytic.s==0]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="red");
  arrows(x0=2, 
         y0=mean(rea2[Analytic.s==0])-se(rea2[Analytic.s==0]), 
         y1=mean(rea2[Analytic.s==0])+se(rea2[Analytic.s==0]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="red");
  arrows(x0=2, 
         y0=mean(fal2[Analytic.s==1])-se(fal2[Analytic.s==1]), 
         y1=mean(fal2[Analytic.s==1])+se(fal2[Analytic.s==1]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="blue");
  arrows(x0=2, 
         y0=mean(rea2[Analytic.s==1])-se(rea2[Analytic.s==1]),
         y1=mean(rea2[Analytic.s==1])+se(rea2[Analytic.s==1]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="blue");  }

# Figure1() draws a line graph of the means of the studied measures across time 
# (before and after manipulation), and argument type.

Figure1 <- function(var1, var2, paneltitle, ylabel, ylimits) {
  # Lines showing means
  plot(x, c(mean(var1[Analytic.s==0]),  mean(var2[Analytic.s==0])), 
       type = "o", col="red", lty=1,
       lwd=3, pch=0, cex=1.5, xlab="Time", 
       ylab=ylabel, main=paneltitle, xaxt="none", xlim=c(0.9, 2.1), 
       ylim=ylimits); 
  axis(1, at=c(1,2), labels=c("Before", "After"));
  lines(x, c(mean(var1[Analytic.s==1]), 
             mean(var2[Analytic.s==1])), 
        type = "o", col="blue",lty=1,
        lwd=3, pch=2, cex=1.5);
  # Error bars in the before condition
  arrows(x0=1, 
         y0=mean(var1[Analytic.s==0])-se(var1[Analytic.s==0]), 
         y1=mean(var1[Analytic.s==0])+se(var1[Analytic.s==0]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="red");
  arrows(x0=1, 
         y0=mean(var1[Analytic.s==1])-se(var1[Analytic.s==1]), 
         y1=mean(var1[Analytic.s==1])+se(var1[Analytic.s==1]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="blue");
  # Error bars in the after condition
  arrows(x0=2, 
         y0=mean(var2[Analytic.s==0])-se(var2[Analytic.s==0]), 
         y1=mean(var2[Analytic.s==0])+se(var2[Analytic.s==0]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="red");
  arrows(x0=2, 
         y0=mean(var2[Analytic.s==1])-se(var2[Analytic.s==1]), 
         y1=mean(var2[Analytic.s==1])+se(var2[Analytic.s==1]), 
         length=0.1, angle=90, code=3, lwd=2.5, col="blue");  }


# The following 17 lines draw Figure 1 with 4 panels.
par(mfrow=c(2,2))
Figure1a(error_fal_1, error_fal_2, error_rea_1, error_rea_2, 
        "(A) Error rate", "Error rate (0-1)", c(0.04,0.27))
legend(1.3, 0.27, # location within Panel A
       c("Intuitive, fallacy", "Intuitive, reasonable", 
         "Analytic, fallacy", "Analytic, reasonable"),
       lwd=3, col=c("red", "red", "blue", "blue"), lty=c(1,2,1,2),
       pch=c(0,1,2,6), pt.cex=1.5, text.font=1)
Figure1(RT1.s, RT2.s, "(B) log(RT)", "log(RT)", c(8.6, 9.35))
legend(1.35, 9.35, c("Intuitive", "Analytic"), lwd=3, col=c("red", "blue"), 
       lty=c(1,1), pch=c(0,2), pt.cex=1.5, text.font=1)
Figure1(Xrat1.s, Xrat2.s, "(C) X ratio", "Xrat", c(0.06, 0.2)) 
legend(1.35, 0.2, c("Intuitive", "Analytic"), lwd=3, col=c("red", "blue"), 
       lty=c(1,1), pch=c(0,2), pt.cex=1.5, text.font=1)
Figure1(AUC1.s, AUC2.s, "(D) AUC", "AUC",  c(0.1, 0.5))
legend(1.35, 0.5, c("Intuitive", "Analytic"), lwd=3, col=c("red", "blue"), 
       lty=c(1,1), pch=c(0,2), pt.cex=1.5, text.font=1)


# --------------------------- FIGURE 2 -----------------------------------------

# In Figures 2 and 3, correct trials end on the right;
# trials on which the participant made an error, end on the left. 
# Note: For fallacies, 'Error'='Strong'. For reasonable arguments, 'Error'='Weak.'
Xpos2 <- Xpos                               # copy the X positions
Xpos2[d0$error==1] <- Xpos[d0$error==1]*-1  # then reverse them for error trials

# Aggreg_plot() plots average trajectories before and after manipulations using
# different colors. 

Aggreg_plot <- function(group, figtitle) {
  # Before, correct trials
  plot(colMeans(Xpos2[group & T1 & d0$error==0,]),              
       colMeans(Ypos[group & T1 & d0$error==0,]),
       type="l", lwd=2, 
       xlim=c(-1,1), ylim=c(0,1.5),
       col="red",
       main=figtitle,
       xlab="Xpos", ylab="Ypos");
  # After, correct trials
  points(colMeans(Xpos2[group & T2 & d0$error==0,]),                
         colMeans(Ypos[group & T2 & d0$error==0,]),
         type="l", lwd=2, 
         xlim=c(-1,1), ylim=c(0,1.5),
         col="blue");
  # Before, error trials
  points(colMeans(Xpos2[group & T1 & d0$error==1,]),               
         colMeans(Ypos[group & T1 & d0$error==1,]),
         type="l", lwd=2, 
         xlim=c(-1,1), ylim=c(0,1.5),
         col="red");
  # After, error trials
  points(colMeans(Xpos2[group & T2 & d0$error==1,]),              
         colMeans(Ypos[group & T2 & d0$error==1,]),
         type="l", lwd=2, 
         xlim=c(-1,1), ylim=c(0,1.5),
         col="blue");
  rect(xleft=-0.1, xright=0.1, ybottom=0.3, ytop=0.1);
  text(x=0, y = 0.2, "START");
  rect(xleft=-1, xright=-0.7,  ybottom=1.1, ytop=1.3);
  text(x=-0.9, y=1.2, "Error*");
  rect(xleft=0.7, xright=1, ybottom=1.1, ytop=1.3);
  text(x=0.8, y=1.2, "Correct*");
  legend(-0.2, 1.5, # location 
         c("Before", "After"),
         lwd=2, col=c("red", "blue", "red", "blue"), 
         text.font=1)}

# The following produces Figure 2.
# Panel A shows the trajectories before and after the intuitive manipulation,
# Panel B shows the trajectories before and after the analytic manipulation. 
par(mfrow=c(1,2))
Aggreg_plot(Analytic==0, "Intuitive")
Aggreg_plot(Analytic==1, "Analytic")


# ----------------------------- FIGURE 3 ---------------------------------------

# Before_after_indiv() plots the actual trajectories for all the trials of one 
# participant, with trials before manipulation in the left panel, and after 
# manipulation in the right panel.

# Used to illustrate the results, using individual participants as examples.

Before_after_indiv <- function(v, manip) {
  Vmat1 <- Xpos2[S==v & T1,]; 
  Ymat1 <- Ypos[S==v & T1,];
  Vmat2 <- Xpos2[S==v & T2,]; 
  Ymat2 <- Ypos[S==v & T2,];
  # Left panel: trials before manipulation
  plot(x= Vmat1[1,], # first trial 
       y= Ymat1[1,],
       xlim=c(-1,1),
       ylim=c(0,1.5),
       typ="l",
       main=paste("Before", manip, "manipulation"),
       xlab="",
       ylab="");
  for (i in 2:nrow(Vmat1)) { # other trials
    points(x= Vmat1[i,],
           y= Ymat1[i,],
           typ="l")}; 
  rect(xleft=-0.1, xright=0.1, ybottom=0.3, ytop=0.1);
  text(x=0, y = 0.2, "START");
  rect(xleft=-1, xright=-0.7,  ybottom=1.1, ytop=1.3);
  text(x=-0.9, y=1.2, "Error*");
  rect(xleft=0.7, xright=1, ybottom=1.1, ytop=1.3);
  text(x=0.8, y=1.2, "Correct*");
  # Right panel: trials after manipulation
  plot(x= Vmat2[1,],       
       y= Ymat2[1,],
       xlim=c(-1,1),
       ylim=c(0,1.5),
       typ="l",
       main=paste("After", manip, "manipulation"),
       xlab="",
       ylab="");
  for (i in 2:nrow(Vmat2)) {    
    points(x= Vmat2[i,],
           y= Ymat2[i,],
           typ="l")};  
  rect(xleft=-0.1, xright=0.1, ybottom=0.3, ytop=0.1);
  text(x=0, y = 0.2, "START");
  rect(xleft=-1, xright=-0.7,  ybottom=1.1, ytop=1.3);
  text(x=-0.9, y=1.2, "Error*");
  rect(xleft=0.7, xright=1, ybottom=1.1, ytop=1.3);
  text(x=0.8, y=1.2, "Correct*")}  


# Give it two subject numbers and indicate which group the persons are from. 
# The following three lines produce Figure 3.
par(mfrow=c(2,2))
Before_after_indiv(58, "intuitive") 
Before_after_indiv(51, "analytic")    

# Here you can check who is in which group. 0 = intuitive, 1 = analytic.
# > Analytic.s
# 1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 
# 0  0  1  1  0  0  1  1  0  0  1  1  0  0  1  1  0  0  1  1 
#
# 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 
# 0  0  1  1  0  0  1  1  0  0  1  1  0  0  1  1  0  0  1  1 
#
# 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 
# 0  0  1  1  0  0  1  1  0  0  1  1  0  0  1  1  0  0


